' Amazin.bas -- Create and Solve Random Mazes
' Rev 1.0.0 William M Leue 7/26/2020
  
  option default integer


  const HSTEP = 4
  const VSTEP = 2
  const CWID  = 3
  const XSTART = 10
  const YSTART = 1
  const EXIT_LIST = 4
  const TRUE = 1
  const FALSE = 0
  
  const USABLE = 1
  const BLOCKED = 0
  const USED = 2
    
  const eLEFT =   0
  const eTOP =    1
  const eRIGHT =  2
  const eBOTTOM = 3
  
  const H = 0
  const V = 1
  
  const HCORR = 1.0    ' change to 0.75 if 16:9 monitor with no letterboxing
  const CELLSIZE = 12
  const LMARGIN = 50
  const TMARGIN = 50
  const DRADIUS = int(cellsize*0.167)
  const WCOLOR = RGB(WHITE)
  const DCOLOR = RGB(RED)
  const ECOLOR = RGB(YELLOW)
  const MCOLOR = RGB(GREEN)
  const NCOLOR = RGB(BLACK)

  const ALLWALLS = 1111
  const UNVISITED = 0
  const VISITED = 1

  ' Global Variables    
  hsize = (800 - HCORR*2*LMARGIN)\(HCORR*CELLSIZE) - 1
  vsize = (600 - 2*TMARGIN)\CELLSIZE - 1
  stack_size = hsize*vsize
  dim matrix(vsize, hsize, 2)
  
  ' Stack of locations for backing out of dead ends
  dim stack(stack_size, 2)
  dim stkptr = 0
  dim decolor = ECOLOR
  
  ' Main program
  text 10, 0, "Creating the Maze..."
  MakeMaze
  DrawMaze
  text 10, 0, "                     "
  text 10, 0, "Running the Maze..."
  RunMaze
  text 10, 0, "                     "
  text 10, 0, "Showing the Cleaned-up Path"
  ShowFinalPath
  end

' Make a new random maze using a depth-first search. This maze will have no loops
sub MakeMaze
  local row, col
  local neighbors(5)
  local wall, nrow, ncol, numunv
  ClearMaze
  stkptr = 0

  ' Starting Cell at Top Left
  row = 0
  col = 0
  MarkCellVisited row, col
  KillWall row, col, 1
  PushCell row, col
  
  ' Depth-first linking the rest of the cells
  do while stkptr > 0
    MarkCellVisited row, col
    GetUnvisitedNeighbors row, col, neighbors() 
    numunv = neighbors(0)
    if numunv > 0 then
      wall = -1
      do
        wall = int(rnd()*4)
        if neighbors(wall+1) = UNVISITED then exit do
      loop
      KillWalls row, col, wall, nrow, ncol
      if nrow < 0 or ncol < 0 or nrow > vsize-1 or ncol > hsize-1 then
        ERROR "matrix coords out of range: nrow = " + str$(nrow) + " ncol = " + str$(ncol)
      end if
      PushCell nrow, ncol
      row = nrow : col = ncol
    else
      do while stkptr > 0
        PopCell nrow, ncol
        GetUnvisitedNeighbors nrow, ncol, neighbors()
        numunv = neighbors(0)
        if numunv > 0 then
          row = nrow : col = ncol
          exit do
        end if
      loop
    end if        
  loop
  KillWall vsize-1, hsize-1, 3

end sub

' Set the maze for creation
sub ClearMaze
  local row, col
  for row = 0 to vsize-1
    for col = 0 to hsize-1
      matrix(row, col, 0) = ALLWALLS
      matrix(row, col, 1) = UNVISITED
    next col
  next row
end sub

' do what it says
sub MarkCellVisited row, col
  matrix(row, col, 1) = VISITED
end sub
        

' return a list of neighboring cells visited status
sub GetUnvisitedNeighbors row, col, neighbors()
  local i, ucount
  ucount = 0
  for i = 1 to 4 : neighbors(i) = VISITED : next i
  if col > 0 then
    if matrix(row,col-1, 1) = UNVISITED then 
      neighbors(1) = UNVISITED
      ucount = ucount + 1
    end if
  end if    
  if row > 0 then
    if matrix(row-1, col, 1) = UNVISITED then 
      neighbors(2) = UNVISITED
      ucount = ucount + 1
    end if
  end if
  if col < hsize-1 then
    if matrix(row, col+1, 1) = UNVISITED then 
      neighbors(3) = UNVISITED
      ucount = ucount + 1
    end if
  end if
  if row < vsize-1 then
    if matrix(row+1, col, 1) = UNVISITED then 
      neighbors(4) = UNVISITED
      ucount = ucount + 1
    end if
  end if
  neighbors(0) = ucount
end sub 


' return the matching wall index for a neighbor's wall
function GetMatchingWall(wall)
  nwall = -1
  select case wall
    case 0
      nwall = 2
    case 1
      nwall = 3
    case 2
      nwall = 0
    case 3
      nwall = 1
  end select
  GetMatchingWall = nwall
end function

' kill the walls between a specified cell and its neighbor
' Be mindfull of maze edges.
sub KillWalls row, col, wall, nrow, ncol
  KillWall row, col, wall
  nrow = -1 : ncol = -1
  select case wall
    case 0
      if col > 0 then 
        nrow = row
        ncol = col - 1
      end if
    case 1
      if row > 0 then 
        nrow = row - 1
        ncol = col
      end if
    case 2
      if col < hsize-1 then 
        nrow = row
        ncol = col + 1
      end if
    case 3
      if row < vsize-1 then 
        nrow = row + 1
        ncol = col
      end if
  end select
  if nrow >= 0 and ncol >= 0 and nrow <= vsize-1 and ncol <= hsize-1 then
    nwall = GetMatchingWall(wall)
    KillWall nrow, ncol, nwall
  end if
end sub
 
' Remove the specified wall from the specified cell
' and its specified neighboring cell
sub KillWall row, col, wall
  local cell, wcode
  local left, top, right, bottom
  
  cell = matrix(row, col, 0)
  left = cell\1000
  top = (cell - left*1000)\100
  right = (cell - left*1000 - top*100)\10
  bottom = (cell - left*1000 - top*100 - right*10)
  select case wall
    case 0
      left = 0
    case 1
      top = 0
    case 2
      right = 0
    case 3
      bottom = 0
  end select
  cell = 1000*left + 100*top + 10*right + bottom 
  matrix(row, col, 0) = cell
end sub

' Draw the maze from the matrix values 
Sub DrawMaze
  cls
  local row, col, cell
  local x1, y1, x2, y2, side, wall
  for row = 0 to vsize-1
    y1 = TMARGIN + row*CELLSIZE
    y2 = y1 + CELLSIZE
    for col = 0 to hsize-1
      x1 = HCORR*LMARGIN + HCORR*col*CELLSIZE 
      x2 = x1 + HCORR*CELLSIZE
      cell = matrix(row, col, 0)
      for side = 0 to 3
        wall = GetWall(cell, side)  
        if wall = 1 then
          select case side
            case 0
              line x1, y1, x1, y2,, WCOLOR
            case 1
              line x1, y1, x2, y1,, WCOLOR
            case 2
              line x2, y1, x2, y2,, WCOLOR
            case 3
              line x1, y2, x2, y2,, WCOLOR
          end select
        end if
      next side
    next col
  next row 
end sub

' returns 1 if the wall for the specified side
' of the specified cell exists or 0 if not.
function GetWall(cell, side)
  local wall = 0
  local west, north, east, south
  west = cell\1000
  north = (cell - west*1000)\100
  east = (cell - west*1000 - north*100)\10
  south = cell - west*1000 - north*100 - east*10
  select case side
    case 0
      if west = 1 then wall = 1
    case 1
      if north = 1 then wall = 1
    case 2
      if east = 1 then wall = 1
    case 3
      if south = 1 then wall = 1
  end select
  GetWall = wall
end function

' Run the maze by trying every path. 
' After the run, MOST of the blind alleys that the solver visited are erased,
' Leaving the more-or-less optimal run from start to finish.
' NOTE: The maze running algorithm does not handle loops in the maze correctly!  
Sub RunMaze
  local row, col, cell, ecell, i, nrow, ncol
  local indir, exits, numExits
  PrepareMazeForRunning
  row = 0
  col = 0
  MarkDirectionUsed row, col, 1
  stkptr = 0
  indir = 1
  do
    ecell = matrix(row, col, 0)
    DrawDot row, col, DCOLOR
    if row = vsize-1 and col = hsize-1 then
      DrawDot row, col, DCOLOR
      exit do  ' winner!
    end if
    numExits = GetNumExits(ecell)
    if numExits >= 1 then
      getNextDirection row, col, nrow, ncol
      pushCell row, col
      row = nrow
      col = ncol
    else
      DrawDot row, col, decolor
      MarkCellDead row, col
      popCell nrow, ncol
      row = nrow
      col = ncol
    end if
  loop
end sub

' the solver uses the '1' copy of the maze for remembering where it has been
' so that it can return after hitting a blind alley.
sub PrepareMazeForRunning
  local row, col
  for row = 0 to vsize-1
    for col = 0 to hsize-1
      matrix(row, col, 1) = 0
    next col
  next row
end sub

' Returns the number of UNUSED exits from the specified maze cell
Function GetNumExits(ecell)
  local num = 0
  if getWall(ecell, eLEFT) = AVAILABLE then
    num = num + 1
  END IF
  if getWall(ecell, eTOP) = AVAILABLE then
    num = num + 1
  END IF
  if getWall(ecell, eRIGHT) = AVAILABLE then
    num = num + 1
  END IF
  if getWall(ecell, eBOTTOM) = AVAILABLE then
    num = num + 1
  END IF
  GetNumExits = num
end function

' Given a cell, find the first available UNUSED exit and return the coords of the cell
' that is pointed to. Also, mark the chosen exit as USED in the matrix.
sub getNextDirection row, col, nrow, ncol
  local ecell, i, mdirection
  ecell = matrix(row, col, 0)
  for i = 0 to 3
    if getWall(ecell, i) = AVAILABLE then
      mdirection = i
      exit for
    end if
  next i
  select case mdirection
    case eLEFT
      ncol = col - 1
      nrow = row
    case eTOP
      ncol = col
      nrow = row - 1
    case eRIGHT
      ncol = col + 1
      nrow = row
    case eBOTTOM
      ncol = col
      nrow = row + 1
  end select
  MarkDirectionUsed row, col, mdirection                ' from cell
  MarkDirectionUsed nrow, ncol, (mdirection+2) MOD 4    ' to cell
end sub

' Mark a direction for a cell as USED
sub MarkDirectionUsed row, col, mdirection
  local ecell, wall
  ecell = matrix(row, col, 0)
  select case mdirection
    case eLEFT
      wall = 1000
    case eTOP
      wall = 100
    case eRIGHT
      wall = 10
    case eBOTTOM
      wall = 1
  end select
  matrix(row, col, 0) = ecell + wall
end sub

' Mark a cell as being in a dead end path
sub MarkCellDead row, col
  matrix(row, col, 1) = 2
end sub

' Show the solver path after removing most of the dead end
' excursions.
sub ShowFinalPath
  local row, col
  for row = 0 to vsize-1
    for col = 0 to hsize-1
      if matrix(row, col, 1) <> 2 then
        DrawDot(row, col, MCOLOR)
      else 
        DrawDot(row, col, NCOLOR)
      end if
    next col
  next row
end sub

' push the current cell and its exit list onto the stack
sub pushCell row, col
  stack(stkptr, 0) = row
  stack(stkptr, 1) = col
  if stkptr >= STACK_SIZE then
    ERROR "Stack Overflow: stkptr = " + str$(stkptr)
  end if
  stkptr = stkptr + 1
end sub

' pop a single cell off the stack and return its coordinates
sub popCell row, col
  stkptr = stkptr - 1
  if stkptr < 0 then
    ERROR "Stack Underflow: stkptr = " + str$(stkptr) 
  end if
  row = stack(stkptr, 0)
  col = stack(stkptr, 1)
end sub
  
' Clear the stack
sub ClearStack
  stkptr = 0
end sub

' Print the stack (for debugging)
sub PrintStack
  local i
  print #1, "stkptr: " + str$(stkptr)
  for i = 0 to stkptr
    print "  " + str$(i) + ": " + str$(stack(i, 0)) + " " + str$(stack(i, 1))
  next i
end sub  

' Draw or erase a Dot in a Cell    
sub DrawDot(row, col, ccode)
  local x, y, jog
  jog = 0
  if CELLSIZE MOD 2 = 0 then
    jog = 1
  end if
  x = HCORR*LMARGIN + HCORR*CELLSIZE*col + HCORR*CELLSIZE/2 + jog
  y = TMARGIN + CELLSIZE*row + CELLSIZE/2
  circle x, y, DRADIUS,, HCORR, ccode, ccode
end sub
    
